<?xml version="1.0" ?>

<!-- Schematron Quick Fixes processing -->

<xsl:stylesheet
   version="2.0"
   xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
   xmlns:axsl="http://www.w3.org/1999/XSL/TransformAlias"
   xmlns:bxsl="http://www.w3.org/1999/XSL/TransformAlias2"
   xmlns:sqf="http://www.schematron-quickfix.com/validator/process"
   xmlns:oqf="http://www.oxygenxml.com/quickfix"
   xmlns:iso="http://purl.oclc.org/dsdl/schematron"
   xmlns:xs="http://www.w3.org/2001/XMLSchema"
   exclude-result-prefixes="iso sqf">

   <xsl:import href="iso-schematron-message.xsl"/>
   
   <!-- The bxsl must be mapped to axsl in output.-->
   <xsl:namespace-alias stylesheet-prefix="bxsl" result-prefix="axsl"/>
   
   <!-- The name of the variable that will contain the quick fix unique identifier -->
   <xsl:variable name="quickFixUuidVarName" select="'quickFixUuid'"/>
   
   <!-- The URI of the generated skeleton file -->
   <xsl:param name="tempSkeletonFileURI"/>
   
   <!-- Key used to identify the quick fixes ids -->
   <xsl:key name="quickFixes" match="sqf:fix|sqf:group/sqf:fix" use="@id"/>
   <xsl:key name="quickFixesGroup" match="sqf:group" use="@id"/>
   
   <!-- Key used to identify the global quick fixes ids -->
   <xsl:key name="globalQuickFixes" match="sqf:fixes/sqf:fix|sqf:fixes/sqf:group/sqf:fix" use="@id"/>
   <xsl:key name="globalQuickFixesGroup" match="sqf:fixes/sqf:group" use="@id"/>
   
   <!-- Overridden to add a namespace alias declaration -->
   <xsl:template name="process-prolog">
      <axsl:output method="xml"/>
      <axsl:namespace-alias stylesheet-prefix="axsl" result-prefix="xsl"/>
   </xsl:template>
   
   <!-- Overridden to add the quick fixes root. -->
   <xsl:template name="process-root">
      <xsl:param name="contents"/>
      <xsl:param name="id" />
      <xsl:param name="version" />
      <xsl:param name="schemaVersion" />
      <xsl:param name="queryBinding" />
      <xsl:param name="title" />
      
      
      <!-- "Rich" parameters -->
      <xsl:param name="fpi" />
      <xsl:param name="icon" />
      <xsl:param name="lang" />
      <xsl:param name="see" />
      <xsl:param name="space" />
      
      <oqf:fixes>
         <xsl:copy-of select="$contents"/>
      </oqf:fixes>
   </xsl:template>
   
   <!-- Overridden to generate a variable for the current rule -->
   <xsl:template name="process-rule">
      <xsl:param name="context" />
      
      <xsl:param name="id" />
      <xsl:param name="flag" />
      
      <!-- "Linkable" parameters -->
      <xsl:param name="role"/>
      <xsl:param name="subject"/>
      
      <!-- "Rich" parameters -->
      <xsl:param name="fpi" />
      <xsl:param name="icon" />
      <xsl:param name="lang" />
      <xsl:param name="see" />
      <xsl:param name="space" />
      
      <xsl:text>&#10;      </xsl:text>
      <xsl:comment>Declare the unique quick fix ID variable <xsl:value-of select="@role" /> </xsl:comment><xsl:text>&#10;</xsl:text>
      <axsl:param name="{oqf:getExecLateFixParamName(current())}" tunnel="yes"/>
      <axsl:param name="{oqf:getUserEntryParamName(current())}" tunnel="yes"/>
      <axsl:param name="{oqf:getLateFixParamName(current())}" tunnel="yes"/>
      <axsl:variable name="{$quickFixUuidVarName}" select="concat(generate-id(), '_M', {count(parent::node()/preceding-sibling::*)})"/>
   </xsl:template>
   
   <!-- At the end of the rule, process the quick fixes referred in rule assert and report elements -->
   <xsl:template name="end-process-rule">
      <!-- Collect all the quick fix ids from the assert and report -->
      <xsl:variable name="assertQuickFixIDs">
         <xsl:for-each select="iso:assert/@sqf:fix">
            <xsl:value-of select="current()"/>
            <xsl:text> </xsl:text>
         </xsl:for-each>
         <xsl:for-each select="iso:report/@sqf:fix">
            <xsl:value-of select="current()"/>
            <xsl:text> </xsl:text>
         </xsl:for-each>
      </xsl:variable>
      
      <!-- Get only the distinct values of the quick fix ids -->
      <xsl:variable name="quickFixIds" select="distinct-values(tokenize($assertQuickFixIDs, '\s'))"/>
      
      <xsl:text>&#10;&#10;</xsl:text>
      <xsl:comment>QUICK FIXES <xsl:value-of select="@role" /> </xsl:comment><xsl:text>&#10;</xsl:text>
      <!-- Process the quick fixes -->
      <xsl:call-template name="process-quick-fixes">
         <xsl:with-param name="checks" select="iso:assert|iso:report"/>
         <xsl:with-param name="quickFixIds" select="$quickFixIds"/>
         <xsl:with-param name="quickFixUuidVarName" select="$quickFixUuidVarName"/>
      </xsl:call-template>
   </xsl:template>

   <!-- Override to generate the quick fix ids in the message element. -->
   <xsl:template name="process-message-end">
      <!-- Set the fix attribute values, the quick fix unique ids -->
      <xsl:if test="@sqf:fix">
         <axsl:text>&#10;Fixes:</axsl:text>
         <xsl:variable name="root" select="root(current())"/>
         <xsl:variable name="ruleNode" select="parent::node()"/>
         <xsl:variable name="assertNode" select="current()"/>
         <xsl:for-each select="distinct-values(tokenize(string(@sqf:fix), '\s'))">
            <xsl:choose>
               <xsl:when test="key('quickFixesGroup', current(), $ruleNode)">
                  <xsl:variable name="fixNodes" select="key('quickFixesGroup', current(), $ruleNode)/sqf:fix"/>
                  <xsl:for-each select="$fixNodes">
                     <xsl:call-template name="generate-id-value">
                        <xsl:with-param name="idVal" select="@id"/>
                     </xsl:call-template>
                  </xsl:for-each>
               </xsl:when>
               <xsl:when test="key('globalQuickFixesGroup', current(), $root)">
                  <xsl:variable name="fixNodes" select="key('globalQuickFixesGroup', current(), $root)/sqf:fix"/>
                  <xsl:for-each select="$fixNodes">
                     <xsl:call-template name="generate-id-value">
                        <xsl:with-param name="idVal" select="@id"/>
                     </xsl:call-template>
                  </xsl:for-each>
               </xsl:when>
               <xsl:when test="key('quickFixes', current(), $ruleNode) | key('globalQuickFixes', current(), $root)">
                  <xsl:call-template name="generate-id-value">
                     <xsl:with-param name="idVal" select="current()"/>
                  </xsl:call-template>
               </xsl:when>
               <xsl:otherwise>
                  <xsl:apply-templates select="$assertNode" mode="errorMsg">
                     <xsl:with-param name="message">
                        Error: Invalid quick fix id reference "<xsl:value-of select="current()"/>".
                     </xsl:with-param>
                  </xsl:apply-templates>
               </xsl:otherwise>
            </xsl:choose>
         </xsl:for-each>
      </xsl:if>
   </xsl:template>
   
   <!-- Show an error message on the selected element from context -->
   <xsl:template match="*" mode="errorMsg">
      <xsl:param name="message" required="yes"/>
      <xsl:message terminate="no"><xsl:value-of select="$message"/></xsl:message>
   </xsl:template>
   
   <!-- Generate a value-of with the unique ID of the fix. -->
   <xsl:template name="generate-id-value">
      <xsl:param name="idVal"/>
      <axsl:value-of>
         <xsl:attribute name="select">
            <xsl:value-of
               select="concat('concat(''', $idVal, '_'', $', $quickFixUuidVarName, ', '' '')')"/>
         </xsl:attribute>
      </axsl:value-of>
   </xsl:template>

   <!-- Function used to create an unique identifier pattern -->
   <xsl:function name="oqf:getQuickFixUniqueId">
      <xsl:param name="quickFixName" required="yes"/>
      <xsl:value-of select="concat($quickFixName, '_{$', $quickFixUuidVarName,'}')"/>
   </xsl:function>
   
   <!-- Process the assertion quick fixes -->
   <xsl:template name="process-quick-fixes">
      <xsl:param name="checks" required="yes"/> <!-- The assert and report nodes from the context -->
      <xsl:param name="quickFixIds" required="yes"/>
      <xsl:param name="quickFixUuidVarName" required="yes"/>
      <xsl:variable name="ruleNode" select="current()"/>
      <xsl:variable name="root" select="root(current())"/>
      
      <xsl:for-each select="$quickFixIds">
         <!-- Get the fix nodes declared local or global -->
         <xsl:variable name="currentFix" select="current()"/>
         <!-- Get the current only the assert/reports that refers the current fix. -->
         <xsl:variable name="currentChecks" select="$checks[tokenize(@sqf:fix, '\s') = $currentFix]"/>
         <xsl:variable name="fixNodes" select="key('quickFixes', $currentFix, $ruleNode)
            | key('globalQuickFixes', current(), $root)"/>
         <xsl:choose>
            <!-- A group of fixes are referred -->
            <xsl:when test="key('quickFixesGroup', current(), $ruleNode)">
               <xsl:variable name="fixNodesLocalGroup" select="key('quickFixesGroup', current(), $ruleNode)/sqf:fix"/>
               <xsl:for-each select="$fixNodesLocalGroup">
                  <!-- Generate the quick-fix for each fix from the group.-->
                  <xsl:call-template name="process-quick-fix">
                     <xsl:with-param name="checks" select="$currentChecks"/>
                     <xsl:with-param name="quickFixNode" select="current()" as="node()"/>
                     <xsl:with-param name="quickFixId" select="oqf:getQuickFixUniqueId(@id)"/>
                     <xsl:with-param name="ruleNode" select="$ruleNode"/>
                  </xsl:call-template>
               </xsl:for-each>
            </xsl:when>
            <xsl:when test="key('globalQuickFixesGroup', current(), $root)">
               <xsl:variable name="fixNodesGlobalGroup" select="key('globalQuickFixesGroup', current(), $root)/sqf:fix"/>
               <xsl:for-each select="$fixNodesGlobalGroup">
                  <!-- Generate the quick-fix for each fix from the group.-->
                  <xsl:call-template name="process-quick-fix">
                     <xsl:with-param name="checks" select="$currentChecks"/>
                     <xsl:with-param name="quickFixNode" select="current()" as="node()"/>
                     <xsl:with-param name="quickFixId" select="oqf:getQuickFixUniqueId(@id)"/>
                     <xsl:with-param name="ruleNode" select="$ruleNode"/>
                  </xsl:call-template>
               </xsl:for-each>
            </xsl:when>
            <xsl:when test="count($fixNodes) > 0">
               <!-- A fix is refereed - generate the quick-fix -->
               <xsl:call-template name="process-quick-fix">
                  <xsl:with-param name="checks" select="$currentChecks"/>
                  <xsl:with-param name="quickFixNode" select="$fixNodes[1]" as="node()"/>
                  <xsl:with-param name="quickFixId" select="oqf:getQuickFixUniqueId(current())"/>
                  <xsl:with-param name="ruleNode" select="$ruleNode"/>
               </xsl:call-template>
            </xsl:when>
         </xsl:choose>
      </xsl:for-each>
   </xsl:template>
   
   <!-- Process the current quick fix -->
   <xsl:template name="process-quick-fix">
      <xsl:param name="checks" required="yes"/> <!-- The assert and report nodes that refers the current quick fix -->
      <xsl:param name="quickFixNode" required="yes"/>
      <xsl:param name="quickFixId" required="yes"/>
      <xsl:param name="ruleNode" required="yes"/>
      
      <!-- Check if is a late quick fix execution, and if the current fix must be executed.  -->
      <xsl:variable name="lateFixParamName" select="oqf:getLateFixParamName($ruleNode)"/>
      <axsl:if test="not(${$lateFixParamName}) or (${$lateFixParamName} = '{$quickFixNode/@id}')">
         <!-- Check if the fix should be used or not for the current context. -->
         <!-- Verify also the assert and report tests -->
         <axsl:if test="{if ($quickFixNode/@use-when) then $quickFixNode/@use-when else 'true()'}
            and ({string-join(
                     for $k in $checks return 
                        if (local-name($k)='report') 
                         then $k/@test 
                         else concat('not(', $k/@test, ')')
                    , ' or ')})">
            <xsl:copy-of select="$quickFixNode/@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
            <axsl:text><xsl:value-of select="'&#10;'"/></axsl:text>
            <oqf:fix id="{$quickFixId}">
               <xsl:copy-of select="$quickFixNode/@role"/>
               <!-- Generate the SQF parameters defined in the quick fix. -->
               <xsl:call-template name="generate-params">
                  <xsl:with-param name="fixNode" select="$quickFixNode"/>
               </xsl:call-template>
               
               <!-- Process the add, remove, replace and string replace instructions -->
               <xsl:call-template name="process-user-entry-and-activity-elements">
                  <xsl:with-param name="fixNode" select="$quickFixNode"/>
                  <xsl:with-param name="ruleNode" select="$ruleNode"/>
                  <xsl:with-param name="lateFixParamName" select="$lateFixParamName"/>
               </xsl:call-template>
            </oqf:fix>
         </axsl:if>
      </axsl:if>
   </xsl:template>
   
   <!-- Process the activity elements from the quick fix, or the call-fix -->
   <xsl:template name="process-user-entry-and-activity-elements">
      <xsl:param name="fixNode" required="yes"/>
      <xsl:param name="ruleNode" required="yes"/>
      <xsl:param name="lateFixParamName" required="yes"/>
      <xsl:variable name="userEntryNode" select="$fixNode/sqf:user-entry"/>
      <!-- Depending if the fix has is late or not, the execution will be done later or now -->
      <!-- The current fix has an user entry => generate or execute a late operation-->
      <xsl:variable name="userEntriesParamName" select="oqf:getUserEntryParamName($ruleNode)"/>
      <xsl:variable name="execLateFixParamName" select="oqf:getExecLateFixParamName($ruleNode)"/>
      <axsl:choose>
         <!-- If the global parameter value is not set, generate a late fix to ask for the user-entry value -->
         <axsl:when test="not(${$execLateFixParamName})">
            <!-- Generate description -->
            <xsl:for-each select="$fixNode/child::node()">
               <xsl:choose>
                  <xsl:when test="namespace-uri() = 'http://www.schematron-quickfix.com/validator/process'">
                     <xsl:if test="local-name() = 'call-fix'">
                        <xsl:apply-templates select="current()" mode="generateCallFixParams"/>
                     </xsl:if>
                     <xsl:if test="local-name() = 'description'">
                        <!-- Generate quick fix description -->
                        <xsl:call-template name="generate-quick-fix-description">
                           <xsl:with-param name="description" select="current()"/>
                        </xsl:call-template>
                     </xsl:if>
                  </xsl:when>
                  <xsl:otherwise>
                     <!-- Other namespaces - maybe an xsl variable declaration or xsl processing -->
                     <xsl:apply-templates mode="sqfProcessing" select="current()"/>
                  </xsl:otherwise>
               </xsl:choose>
            </xsl:for-each>
            
           <!-- Get the first operation type that can can be executed -->
            <xsl:variable name="operationType">
               (<xsl:apply-templates select="$fixNode/child::node()" mode="getFirstValidOperationType"/>'noOperation')[1]
            </xsl:variable>
            <axsl:variable name="firstValidOp" select="{normalize-space($operationType)}"/>
            <axsl:if test="$firstValidOp != 'noOperation'">
               <!-- Generate late execution operation-->
               <oqf:lateOperation>
                  <!-- Set the current operations role. -->
                  <xsl:for-each select="$userEntryNode">
                     <xsl:variable name="pos" select="position() - 1"/>
                     <xsl:attribute name="oqf:userEntryTitle{$pos}" select="sqf:description/sqf:title/text()"/>
                     <xsl:attribute name="oqf:userEntryDefault{$pos}" select="@default"/>
                     <xsl:attribute name="oqf:userEntryType{$pos}" select="@type"/>
                  </xsl:for-each>
                  <xsl:attribute name="schemaSystemId" select="base-uri($fixNode)"/>

                  <axsl:attribute name="role" select="$firstValidOp"/>
                  <axsl:attribute name="oqf:systemId" select="saxon:system-id()" use-when="function-available('saxon:system-id')"/>
                  <xsl:variable name="globalUserEntryParamName" select="'userEntriesValues'"/>
                  <xsl:variable name="globalLateFixParamName" select="'lateQuickFixName'"/>
                  <!-- Generate a stylesheet for the late operation -->
                  <bxsl:stylesheet version="3.0">
                     <!-- Import the skeleton in order to use it to generate the operations later -->
                     <bxsl:import href="{$tempSkeletonFileURI}"/><axsl:text><xsl:value-of select="'&#10;'"/></axsl:text>
                     <!-- Stylesheet global parameter used to pass the user entry value. -->
                     <bxsl:param name="{$globalUserEntryParamName}"/>
                     <!-- Stylesheet global parameter used to pass the quick fix name that needs to be executed. -->
                     <bxsl:param name="{$globalLateFixParamName}" required="yes"/>
                     
                     <!-- The current generated template mode -->
                     <xsl:variable name="templateMode" select="concat('M', count($ruleNode/../preceding-sibling::*))"/>
                     <!-- The current node path -->
                     <axsl:variable name="nodePath">
                        <axsl:apply-templates select="current()" mode="schematron-get-full-path"/>
                     </axsl:variable>
                     <!-- The template that matches the root and calls the template that will generate the operations.  -->
                     <bxsl:template match="/">
                        <oqf:fixes xmlns:oqf="http://www.oxygenxml.com/quickfix">
                           <!-- Apply the template that will generate the operations. -->
                           <bxsl:apply-templates mode="{$templateMode}">
                              <!-- Apply the template over the context node -->
                              <axsl:attribute name="select" select="$nodePath"/>
                              <!--  Template parameter used to check if we are in the execute late fix mode. -->
                              <bxsl:with-param name="{$execLateFixParamName}" tunnel="yes" select="true()"/>
                              <!--  Template parameter used to pass the user entry value. -->
                              <bxsl:with-param name="{$userEntriesParamName}" tunnel="yes" select="${$globalUserEntryParamName}"/>
                              <!-- Stylesheet global parameter used to pass the quick fix name that needs to be executed. -->
                              <bxsl:with-param name="{$lateFixParamName}" tunnel="yes" select="${$globalLateFixParamName}"/>
                           </bxsl:apply-templates>
                        </oqf:fixes>
                     </bxsl:template>
                     
                     <axsl:if test="not(self::comment() | self::processing-instruction() | self::attribute())">
                        <bxsl:template mode="{$templateMode}">
                           <axsl:attribute name="match" select="concat($nodePath, '/node()')"/>
                        </bxsl:template>
                     </axsl:if>
                  </bxsl:stylesheet>
               </oqf:lateOperation>
            </axsl:if>
         </axsl:when>
         <axsl:otherwise>
            <xsl:for-each select="$fixNode/child::node()">
               <xsl:choose>
                  <xsl:when test="namespace-uri() = 'http://www.schematron-quickfix.com/validator/process'">
                     <xsl:choose>
                        <xsl:when test="local-name() = 'description'">
                           <!-- Generate quick fix description -->
                           <xsl:call-template name="generate-quick-fix-description">
                              <xsl:with-param name="description" select="current()"/>
                           </xsl:call-template>
                        </xsl:when>
                        <xsl:when test="local-name() = 'user-entry'">
                           <!-- Create a variable with the content from the user  -->
                           <axsl:variable name="{@name}" select="${$userEntriesParamName}[{count(preceding-sibling::sqf:user-entry) + 1}]"/>
                        </xsl:when>
                        <xsl:otherwise>
                           <!-- Process other SQF elements -->
                           <xsl:call-template name="process-activity-elements"/>
                        </xsl:otherwise>
                     </xsl:choose>
                  </xsl:when>
                  <xsl:otherwise>
                     <!-- Other namespaces - maybe an xsl variable declaration or xsl processing -->
                     <xsl:apply-templates mode="sqfProcessing" select="current()"/>
                  </xsl:otherwise>
               </xsl:choose>
            </xsl:for-each>
         </axsl:otherwise>
      </axsl:choose>
   </xsl:template>
   
   <!-- Generate first valid operation type -->
   <xsl:template match="sqf:add | sqf:delete | sqf:replace | sqf:stringReplace" mode="getFirstValidOperationType">
      .<xsl:if test="@use-when">[<xsl:value-of select="@use-when"/>]</xsl:if>
      <xsl:if test="@match">[<xsl:value-of select="@match"/>]</xsl:if>/'<xsl:value-of select="local-name()"/>',
   </xsl:template>
   
   <!-- Template used to generate the first valid opration from a call fix. -->
   <xsl:template match="sqf:call-fix" mode="getFirstValidOperationType">
      <xsl:variable name="fixNode" select="oqf:getCalledFixNode(current())"/>
      <xsl:if test="$fixNode">
         <xsl:apply-templates select="$fixNode/child::node()" mode="getFirstValidOperationType"/>
      </xsl:if>
   </xsl:template>
   
      
   <!-- Template that matches the other elements that are not treated when in 'getFirstValidOperationType' mode -->
   <xsl:template match="*" mode="getFirstValidOperationType"/>
   
    <!-- Template used to generate the call fix parameters and variables recursively. -->
   <xsl:template match="sqf:call-fix" mode="generateCallFixParams">
      <xsl:variable name="fixNode" select="oqf:getCalledFixNode(current())"/>
      <xsl:if test="$fixNode">
         <xsl:call-template name="generate-params">
            <xsl:with-param name="fixNode" select="$fixNode"/>
            <xsl:with-param name="withParams" select="sqf:with-param"/>
            <xsl:with-param name="checkWithParams" select="true()"/>
            <xsl:with-param name="callFixNode" select="current()"/>
            <xsl:with-param name="showErr" select="false()"/>
         </xsl:call-template>
         <xsl:apply-templates select="$fixNode/child::node()[namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']" mode="sqfProcessing"/>
         <xsl:apply-templates select="$fixNode/sqf:call-fix" mode="generateCallFixParams"/>
      </xsl:if>
   </xsl:template>
   <xsl:template match="*" mode="generateCallFixParams"/>
   
   <!-- Function used to get the called fix node -->
   <xsl:function name="oqf:getCalledFixNode">
      <xsl:param name="callFixNode" as="node()"/>
      <xsl:variable name="refFix" select="$callFixNode/@ref"/>
      <!-- Get the referenced fix -->
      <xsl:variable name="ruleNode" select="$callFixNode/ancestor::iso:rule"/>
      <xsl:variable name="root" select="root($callFixNode)"/>
      <xsl:variable name="fixNodes" select="
            (if ($ruleNode) then
               key('quickFixes', $refFix, $ruleNode)
            else
               $ruleNode)
            | key('globalQuickFixes', $refFix, $root)"
      />
      <xsl:sequence select="if (count($fixNodes) > 0) then $fixNodes[1] else $fixNodes"/>
   </xsl:function>
   
   <!-- Function used to generate the execLateFix parameter name -->
   <xsl:function name="oqf:getExecLateFixParamName">
      <xsl:param name="ruleNode" required="yes"/>
      <xsl:value-of select="concat('execLateFix_', generate-id($ruleNode/parent::node()))"/>
   </xsl:function>
   
   <!-- Function used to generate the user-entry parameter name -->
   <xsl:function name="oqf:getUserEntryParamName">
      <xsl:param name="ruleNode" required="yes"/>
      <xsl:value-of select="concat('userEntriesValues_', generate-id($ruleNode/parent::node()))"/>
   </xsl:function>
   
   <!-- Function used to generate the late quick fix parameter name -->
   <xsl:function name="oqf:getLateFixParamName">
      <xsl:param name="ruleNode" required="yes"/>
      <xsl:value-of select="concat('lateFix_', generate-id($ruleNode/parent::node()))"/>
   </xsl:function>
   
   <!-- Process the activity elements from the quick fix, or the call-fix -->
   <xsl:template name="process-activity-elements">
      <xsl:choose>
         <!-- Process the quick fix nodes -->
         <xsl:when test="namespace-uri() = 'http://www.schematron-quickfix.com/validator/process'">
            <xsl:choose>
               <!-- Process call-fix -->
               <xsl:when test="local-name() = 'call-fix'">
                  <xsl:call-template name="process-call-fix"/>
               </xsl:when>
               <!-- Process Add operation -->
               <xsl:when test="local-name() = 'add'">
                  <axsl:if test="{if (@use-when) then @use-when  else 'true()'}">
                     <xsl:call-template name="process-add-operation"/>
                  </axsl:if>
               </xsl:when>
               <!-- Process Delete operation -->
               <xsl:when test="local-name() = 'delete'">
                  <axsl:if test="{if (@use-when) then @use-when  else 'true()'}">
                     <xsl:call-template name="process-delete-operation"/>
                  </axsl:if>
               </xsl:when>
               <!-- Process StringReplace Operation -->
               <xsl:when test="local-name() = 'stringReplace'">
                  <axsl:if test="{if (@use-when) then @use-when  else 'true()'}">
                     <xsl:call-template name="process-string-replace-operation"/>
                  </axsl:if>
               </xsl:when>
               <!-- Process Replace operation -->
               <xsl:when test="local-name() = 'replace'">
                  <axsl:if test="{if (@use-when) then @use-when  else 'true()'}">
                     <xsl:call-template name="process-replace-operation"/>
                  </axsl:if>
               </xsl:when>
            </xsl:choose>
         </xsl:when>
         <!-- Other namespaces - process them -->
         <xsl:otherwise>
            <xsl:apply-templates mode="sqfProcessing" select="current()"/>
         </xsl:otherwise>
      </xsl:choose>
      <axsl:text>
         <xsl:value-of select="'&#10;'"/>
      </axsl:text>
   </xsl:template>
   
   <!-- Generate the description of the quick fix -->
   <xsl:template name="generate-quick-fix-description">
      <xsl:param name="description" required="yes"/>
      <oqf:name>
         <xsl:for-each select="$description/sqf:title">
            <xsl:apply-templates mode="text"/>
         </xsl:for-each>
      </oqf:name>
      <oqf:description>
         <xsl:for-each select="$description/sqf:p">
            <!-- Copy the attributes -->
            <oqf:p>
               <!-- TODO Check if the attributes must be generated. Maybe the xml:lang
                                             but not the oxy attributes-->
               <!--<xsl:apply-templates select="@*"/>-->
               <!-- Process the content of the para as text, the sch:value-of, sch:name, sch:emph, 
                                             sch:dir, sch:span are processed in this mode.-->
               <xsl:apply-templates mode="text"/>
            </oqf:p>
         </xsl:for-each>
      </oqf:description>
      <axsl:text>
         <xsl:value-of select="'&#10;'"/>
      </axsl:text>
   </xsl:template>
   
   <!-- Process the current call-fix element -->
   <xsl:template name="process-call-fix">
      <!-- Get the referenced fix -->
      <xsl:variable name="fixNode" select="oqf:getCalledFixNode(current())"/>
      
      <!-- Check there is an id of the referred quick fix -->
      <xsl:if test="not($fixNode)">
         <xsl:apply-templates select="current()" mode="errorMsg">
            <xsl:with-param name="message">
               Error: Invalid quick fix id reference "<xsl:value-of select="@ref"/>".
            </xsl:with-param>
         </xsl:apply-templates>
      </xsl:if>
      
      <!-- Process the current fix -->
      <xsl:if test="$fixNode and not($fixNode[1]/child::node()[
         namespace-uri() = 'http://www.schematron-quickfix.com/validator/process' and
         local-name() = 'user-entry'])">
         
         <xsl:call-template name="generate-params">
            <xsl:with-param name="fixNode" select="$fixNode"/>
            <xsl:with-param name="withParams" select="sqf:with-param"/>
            <xsl:with-param name="checkWithParams" select="true()"/>
            <xsl:with-param name="callFixNode" select="current()"/>
         </xsl:call-template>
         
         <!-- Process the activity elements from the called template -->
         <xsl:for-each select="$fixNode/child::node()">
            <xsl:call-template name="process-activity-elements"/>
         </xsl:for-each>
      </xsl:if>
   </xsl:template>
   
   <!-- Generate for each parameter a variable with the value from the parameter. -->
   <xsl:template name="generate-params">
      <xsl:param name="fixNode" required="yes"/>
      <xsl:param name="withParams" as="node()*"/>
      <xsl:param name="checkWithParams" as="xs:boolean" select="false()"/>
      <xsl:param name="callFixNode"/>
      <xsl:param name="showErr" as="xs:boolean" select="true()"/>
      
      <!-- Check that all parameters are defined. -->
      <xsl:if test="$showErr">
         <xsl:for-each select="$withParams">
            <xsl:if test="not(@name=$fixNode/child::sqf:param/@name)">
               <xsl:apply-templates select="current()" mode="errorMsg">
                  <xsl:with-param name="message">
                     Error: The parameter "<xsl:value-of select="@name"/>" is not defined in the called fix.
                  </xsl:with-param>
               </xsl:apply-templates>
            </xsl:if>
         </xsl:for-each>
      </xsl:if>
      
      <xsl:for-each select="$fixNode/child::sqf:param">
         <!-- Generate a variable for each parameter -->
         <axsl:variable name="{@name}">
            <xsl:copy-of select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
            <xsl:variable name="refParams" select="if ($withParams) then $withParams[@name = current()/@name] else $withParams"/>
            <xsl:choose>
               <xsl:when test="$checkWithParams and count($refParams) = 0 and @required='yes'">
                  <xsl:if test="$showErr">
                     <xsl:apply-templates select="$callFixNode" mode="errorMsg">
                        <xsl:with-param name="message">
                           Error: Required parameter "<xsl:value-of select="@name"/>" is not set is the call-fix.
                        </xsl:with-param>
                     </xsl:apply-templates>
                  </xsl:if>
               </xsl:when>
               <xsl:when test="count($refParams) > 0">
                  <xsl:if test="$refParams[1]/@select">
                     <xsl:attribute name="select" select="$refParams[1]/@select"/>
                  </xsl:if>
                  <xsl:copy-of select="$refParams[1]/child::node()"/>
               </xsl:when>
               <xsl:otherwise>
                  <xsl:if test="@default">
                     <xsl:attribute name="select" select="@default"/>
                  </xsl:if>
                  <xsl:copy-of select="child::node()"/>
               </xsl:otherwise>
            </xsl:choose>
         </axsl:variable>
      </xsl:for-each>
   </xsl:template>
   
   <!-- Process the replace operation from the current fix -->
   <xsl:template name="process-replace-operation">
      <!-- The match node for the replace operation -->
      <xsl:variable name="match" select="oqf:getCurrentOperationMatch(current())"/>
      
      <axsl:choose>
         <axsl:when test="({$match})/self::text() and {not(@node-type) or @node-type='keep'}()">
            <!-- Replace text with a fragment, that can be also text -->
            <xsl:call-template name="process-string-replace-operation"/>
         </axsl:when>
         <axsl:otherwise>
            <!-- Do not generate the delete operation if you replace an attribute with the same name. -->
            <xsl:variable name="nodeType" select="@node-type"/>
            <!-- Get the target attribute value, convert the AVT to be used in a select -->
            <xsl:variable name="target">
               <xsl:choose>
                  <xsl:when test="@target">
                     concat(''<xsl:analyze-string select="@target" regex="\{{(.+?)\}}">
                        <xsl:matching-substring>,<xsl:value-of select="regex-group(1)"/></xsl:matching-substring>
                        <xsl:non-matching-substring>,'<xsl:value-of select="."/>'</xsl:non-matching-substring>
                     </xsl:analyze-string>)
                  </xsl:when>
                  <xsl:otherwise>''</xsl:otherwise>
               </xsl:choose>
            </xsl:variable>
            
            <axsl:for-each select="{$match}">
               <!-- Perform replace in reverse order -->
               <axsl:sort select="position()" data-type="number" order="descending"/>
               <xsl:call-template name="generate-add-node-operation">
                  <xsl:with-param name="position" select="'after'"/>
               </xsl:call-template>
               
               <axsl:variable name="isNotSameAttrReplace" select="(not({($nodeType='attribute' or $nodeType='@')}()) and not({$nodeType='keep'}() 
                  and self::attribute())) or name()!= {normalize-space($target)}"/>
               <!-- Do not generate the delete operation if you replace an attribute with the same name. -->
               <axsl:if test="$isNotSameAttrReplace">
                  <xsl:copy-of select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
                  <xsl:call-template name="generate-delete-node-operation"/>
               </axsl:if>
            </axsl:for-each>
         </axsl:otherwise>
      </axsl:choose>
      
   </xsl:template>
   
   <!-- Process the string replace operation from the current quick fix. -->
   <xsl:template name="process-string-replace-operation">
      <xsl:text>&#10;</xsl:text>
      
      <xsl:if test="@regex">
         <xsl:comment> Just validate the Regex.</xsl:comment>
         <xsl:text>&#10;</xsl:text>
         <axsl:analyze-string select="." regex="{@regex}">
            <xsl:copy-of select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
            <axsl:matching-substring></axsl:matching-substring>
         </axsl:analyze-string>
      </xsl:if>
      
      <xsl:text>&#10;</xsl:text>
      <xsl:comment> For each context node generate the string operation.</xsl:comment>
      
      <axsl:for-each select="{oqf:getCurrentOperationMatch(current())}">
         <xsl:copy-of select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
         <!-- Perform replace in reverse order -->
         <axsl:sort select="position()" data-type="number" order="descending"/>
         <oqf:changeText>
            <!-- The regular expression. -->
            <xsl:if test="@regex">
               <xsl:attribute name="oqf:regex" select="@regex"/>
            </xsl:if>
            <!-- Set the context -->
            <axsl:attribute name="oqf:context">
               <axsl:apply-templates select="current()" mode="schematron-get-full-path"/>
            </axsl:attribute>
            <axsl:attribute name="oqf:systemId" select="saxon:system-id()" use-when="function-available('saxon:system-id')"/>
            <!-- The new content -->
            <xsl:call-template name="generate-operation-content"/>
         </oqf:changeText>
      </axsl:for-each>
   </xsl:template>
   
   <!-- Process the delete operation from the current quick fix. -->
   <xsl:template name="process-delete-operation">
      <xsl:text>&#10; &#10;</xsl:text>
      <xsl:comment>For each context node generate the delete operation.</xsl:comment>
      <xsl:text>&#10;</xsl:text>
      <axsl:for-each select="{oqf:getCurrentOperationMatch(current())}">
         <xsl:copy-of select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
         <axsl:sort select="position()" data-type="number" order="descending"/>
         <!-- The delete context -->
         <!-- Create the delete operation -->
         <xsl:call-template name="generate-delete-node-operation"/>
      </axsl:for-each>
   </xsl:template>
   
   <!-- Generate the delete node operation depending on the  current node type. -->
   <xsl:template name="generate-delete-node-operation">
      <axsl:choose>
         <axsl:when test="self::attribute()">
            <axsl:variable name="context">
               <xsl:copy-of
                  select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
               <axsl:apply-templates select="parent::node()" mode="schematron-get-full-path"/>
            </axsl:variable>
            <!-- For attribute is a set an attribute with null value -->
            <oqf:setAttribute>
               <axsl:attribute name="oqf:name" select="node-name(current())"/>
               <axsl:attribute name="oqf:context" select="$context"/>
               <axsl:attribute name="oqf:systemId" select="saxon:system-id()"
                  use-when="function-available('saxon:system-id')"/>
               <axsl:attribute name="oqf:delete" select="true()"/>
            </oqf:setAttribute>
         </axsl:when>
         <axsl:when test="self::text()">
            <axsl:variable name="context">
               <xsl:copy-of
                  select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
               <axsl:apply-templates select="current()" mode="schematron-get-full-path"/>
            </axsl:variable>
            <!-- For text nodes, generate a change text node operation -->
            <oqf:changeText>
               <!-- Set the context -->
               <axsl:attribute name="oqf:context" select="$context"/>
               <axsl:attribute name="oqf:systemId" select="saxon:system-id()"
                  use-when="function-available('saxon:system-id')"/>
            </oqf:changeText>
         </axsl:when>
         <axsl:otherwise>
            <axsl:variable name="context">
               <xsl:copy-of
                  select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
               <axsl:apply-templates select="current()" mode="schematron-get-full-path"/>
            </axsl:variable>
            <!-- For ouher nodes, use delete operation and specify the path of the node to be deleted -->
            <oqf:delete>
               <axsl:attribute name="oqf:context" select="$context"/>
               <axsl:attribute name="oqf:systemId" select="saxon:system-id()"
                  use-when="function-available('saxon:system-id')"/>
            </oqf:delete>
         </axsl:otherwise>
      </axsl:choose>
   </xsl:template>
   
   <!-- Process an Add operation from a quick fix -->
   <xsl:template name="process-add-operation">
      <xsl:text>&#10;&#10;</xsl:text>
      <xsl:comment>For each context node generate the add operation.</xsl:comment>
      <xsl:text>&#10;</xsl:text>
      <axsl:for-each select="{oqf:getCurrentOperationMatch(current())}">
         <!-- Perform add in reverse order -->
         <axsl:sort select="position()" data-type="number" order="descending"/>
         <xsl:call-template name="generate-add-node-operation"/>
      </axsl:for-each>
   </xsl:template>
   
   <!-- Generate an add node operation, depending on the cuurent node to be added. -->
   <xsl:template name="generate-add-node-operation">
      <xsl:param name="position" select="@position"/>
      <axsl:variable name="nodeType">
         <xsl:copy-of
            select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
         <xsl:comment>The current node type to be added.</xsl:comment>
         <axsl:choose>
            <axsl:when test="{@node-type='keep'}()">
               <xsl:copy-of
                  select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
               <!-- Detect from the contexy node the type -->
               <axsl:choose>
                  <axsl:when test="self::attribute()">attribute</axsl:when>
                  <axsl:when test="self::comment()">comment</axsl:when>
                  <axsl:when test="self::processing-instruction()">pi</axsl:when>
                  <axsl:otherwise>element</axsl:otherwise>
               </axsl:choose>
            </axsl:when>
            <axsl:otherwise>
               <axsl:value-of select="'{@node-type}'">
                  <xsl:copy-of
                     select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"
                  />
               </axsl:value-of>
            </axsl:otherwise>
         </axsl:choose>
      </axsl:variable>

      <!-- Compute the XPath expression of the current node. -->
      <xsl:comment>Context of the operation</xsl:comment>
      <axsl:variable name="context">
         <xsl:copy-of
            select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
         <axsl:choose>
            <axsl:when test="self::attribute()">
               <!-- For attributes, the context is the parent element. -->
               <axsl:apply-templates select="parent::node()" mode="schematron-get-full-path"/>
            </axsl:when>
            <axsl:otherwise>
               <!-- Compute the XPath expression of the current node. -->
               <axsl:apply-templates select="current()" mode="schematron-get-full-path"/>
            </axsl:otherwise>
         </axsl:choose>
      </axsl:variable>

      <xsl:comment>Insert the content depending on the node type.</xsl:comment>
      <axsl:choose>
         <!-- Add Element -->
         <axsl:when test="$nodeType = 'element'">
            <xsl:copy-of
               select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
            <oqf:insertElement>
               <xsl:copy-of
                  select="@*[local-name() = 'space' and namespace-uri() = 'http://www.w3.org/XML/1998/namespace']"/>
               <!-- The name of the target element -->
               <xsl:attribute name="oqf:name" select="@target"/>
               <!-- Set the insertion context -->
               <axsl:attribute name="oqf:context" select="$context"/>
               <axsl:attribute name="oqf:systemId" select="saxon:system-id()"
                  use-when="function-available('saxon:system-id')"/>

               <!-- Set the insert position -->
               <xsl:if test="$position">
                  <axsl:attribute name="oqf:position" select="'{$position}'"/>
               </xsl:if>

               <!-- Set the content of the element -->
               <xsl:call-template name="generate-operation-content"/>
            </oqf:insertElement>
         </axsl:when>
         <!-- Add Attribute -->
         <axsl:when test="$nodeType = 'attribute' or $nodeType = '@'">
            <xsl:copy-of
               select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
            <oqf:setAttribute oqf:name="{@target}">
               <!-- Set the insertion context -->
               <axsl:attribute name="oqf:context" select="$context"/>
               <axsl:attribute name="oqf:systemId" select="saxon:system-id()"
                  use-when="function-available('saxon:system-id')"/>

               <!-- Set the content of the attribute -->
               <xsl:call-template name="generate-operation-content"/>
            </oqf:setAttribute>
         </axsl:when>
         <axsl:when
            test="not($nodeType) or $nodeType = '' or $nodeType = 'comment' or $nodeType = 'pi' or $nodeType = 'processing-instruction'">
            <xsl:copy-of
               select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
            <oqf:insertFragment>
               <xsl:copy-of
                  select="@*[local-name() = 'space' and namespace-uri() = 'http://www.w3.org/XML/1998/namespace']"/>
               <!-- Set the insertion context -->
               <axsl:attribute name="oqf:context" select="$context"/>
               <axsl:attribute name="oqf:systemId" select="saxon:system-id()"
                  use-when="function-available('saxon:system-id')"/>

               <!-- Set the insert position -->
               <xsl:if test="$position">
                  <axsl:attribute name="oqf:position" select="'{$position}'"/>
               </xsl:if>

               <!-- Set the content of the fragment -->
               <axsl:choose>
                  <axsl:when test="$nodeType = 'comment'">
                     <xsl:copy-of
                        select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
                     <axsl:comment>
                        <!-- Set the content of the comment -->
                        <xsl:call-template name="generate-operation-content"/>
                     </axsl:comment>
                  </axsl:when>
                  <axsl:when test="$nodeType = 'processing-instruction' or $nodeType = 'pi'">
                     <xsl:copy-of
                        select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
                     <axsl:processing-instruction name="{@target}">
                        <!-- Set the content of the PI -->
                        <xsl:call-template name="generate-operation-content"/>
                     </axsl:processing-instruction>
                  </axsl:when>
                  <axsl:otherwise>
                     <!-- Set the content of the fragment -->
                     <xsl:call-template name="generate-operation-content"/>
                  </axsl:otherwise>
               </axsl:choose>
            </oqf:insertFragment>
         </axsl:when>
      </axsl:choose>
   </xsl:template>
   
   <!-- Get the current operation match XPath expression -->
   <xsl:function name="oqf:getCurrentOperationMatch">
      <xsl:param name="operationNode"/>
      <!-- The match attribute is '.', in case is missing -->
      <xsl:choose>
         <xsl:when test="$operationNode/@match">
            <xsl:value-of select="$operationNode/@match"/>
         </xsl:when>
         <xsl:otherwise>.</xsl:otherwise>
      </xsl:choose>
   </xsl:function>
   
   <!-- Get the content to be inserted from the current node or specified by the 'select' attribute path -->
   <xsl:template name="generate-operation-content">
      <xsl:param name="contextNode"/>
      <xsl:choose>
         <xsl:when test="@select">
            <!-- If a select attribute is present, the content will be the content matched by the xpath 
               from 'select' attribute -->
            <axsl:copy-of select="{@select}">
               <xsl:copy-of select="@*[namespace-uri() != '' and namespace-uri() != 'http://www.schematron-quickfix.com/validator/process']"/>
            </axsl:copy-of>
         </xsl:when>
         <xsl:otherwise>
            <!-- The content is defied by the content of the context node. -->
            <xsl:apply-templates select="child::node()" mode="sqfProcessing"/>
         </xsl:otherwise>
      </xsl:choose>
   </xsl:template>
     
  
   <!-- Process the let from the fix, not from paragraph. -->
   <xsl:template match="iso:let" mode="sqfProcessing">
      <xsl:apply-templates select="."/>
   </xsl:template>
   
   <xsl:template match="iso:value-of" mode="sqfProcessing">
      <xsl:apply-templates  select=". " mode="text"/>
   </xsl:template>
   
   <!-- For keep element generate an xsl:copy-of instruction-->
   <xsl:template match="sqf:keep" mode="sqfProcessing">
      <axsl:copy-of select="{@select}"/>
   </xsl:template>
   
   <!-- On the quick fix nodes we don’t add the position info -->
   <xsl:template match="@*[namespace-uri()='http://www.oxygenxml.com/schematron/validation']" mode="sqfProcessing"/>
   
   
   <!-- By default copy all the content in the sqfProcessing mode -->
   <xsl:template match="*|text()|@*" mode="sqfProcessing">
      <xsl:copy copy-namespaces="no">
         <xsl:apply-templates select="@*" mode="sqfProcessing"/>
         <xsl:apply-templates mode="sqfProcessing"/>
      </xsl:copy>
   </xsl:template>
   
   <!-- Skip the processing of the other schematron-quick-fixes nodes -->
   <xsl:template match="sqf:*"/>
</xsl:stylesheet>